<?php
/**
 * Plugin Name: Creative Design SMS Gateway
 * Plugin URI: https://www.creativedesign.com.bd/
 * Description: Official WooCommerce SMS Notification Plugin for Creative Design SMS Gateway.
 * Version: 1.0.0
 * Author: Creative Design
 * Author URI: https://www.creativedesign.com.bd/
 * License: GPL2
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class CreativeDesign_SMS_Plugin {

    public function __construct() {
        // Admin Menu & Settings
        add_action('admin_menu', array($this, 'cd_sms_menu'));
        add_action('admin_init', array($this, 'cd_sms_settings_init'));

        // WooCommerce Order Hooks
        add_action('woocommerce_new_order', array($this, 'send_admin_new_order_sms'), 10, 1);
        add_action('woocommerce_order_status_changed', array($this, 'send_customer_order_status_sms'), 10, 4);
        
        // User Registration Hook (OTP/Welcome)
        add_action('user_register', array($this, 'send_registration_otp_sms'), 10, 1);
    }

    // --- 1. Admin Menu & Settings ---
    public function cd_sms_menu() {
        add_options_page(
            'Creative SMS Settings',
            'Creative SMS',
            'manage_options',
            'creative-sms-gateway',
            array($this, 'cd_sms_settings_page')
        );
    }

    public function cd_sms_settings_init() {
        register_setting('cd_sms_group', 'cd_sms_api_url');
        register_setting('cd_sms_group', 'cd_sms_api_key');
        register_setting('cd_sms_group', 'cd_sms_admin_phone');
        register_setting('cd_sms_group', 'cd_sms_enable_admin');
        register_setting('cd_sms_group', 'cd_sms_enable_customer');
        register_setting('cd_sms_group', 'cd_sms_otp_template');
    }

    public function cd_sms_settings_page() {
        ?>
        <div class="wrap">
            <h1>Creative Design SMS Gateway Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields('cd_sms_group'); ?>
                <?php do_settings_sections('cd_sms_group'); ?>
                
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">API URL</th>
                        <td>
                            <input type="text" name="cd_sms_api_url" value="<?php echo esc_attr(get_option('cd_sms_api_url', 'https://www.creativedesign.com.bd/api/smsapi')); ?>" style="width: 100%;" />
                            <p class="description">Default: https://www.creativedesign.com.bd/api/smsapi</p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">API Key</th>
                        <td><input type="text" name="cd_sms_api_key" value="<?php echo esc_attr(get_option('cd_sms_api_key')); ?>" style="width: 100%;" /></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Admin Phone Number</th>
                        <td><input type="text" name="cd_sms_admin_phone" value="<?php echo esc_attr(get_option('cd_sms_admin_phone')); ?>" placeholder="88017XXXXXXXX" /></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Notifications</th>
                        <td>
                            <label><input type="checkbox" name="cd_sms_enable_admin" value="1" <?php checked(1, get_option('cd_sms_enable_admin'), true); ?> /> Send SMS to Admin on New Order</label><br>
                            <label><input type="checkbox" name="cd_sms_enable_customer" value="1" <?php checked(1, get_option('cd_sms_enable_customer'), true); ?> /> Send SMS to Customer on Order Status Change</label>
                        </td>
                    </tr>
                     <tr valign="top">
                        <th scope="row">Registration Message / OTP</th>
                        <td>
                            <textarea name="cd_sms_otp_template" rows="3" style="width: 100%;"><?php echo esc_attr(get_option('cd_sms_otp_template', 'Welcome! Your OTP is {otp}. Thanks for joining us.')); ?></textarea>
                            <p class="description">Use <b>{otp}</b> variable to insert the verification code.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    // --- 2. Helper Function to Send SMS ---
    private function send_sms($number, $message) {
        $api_url = get_option('cd_sms_api_url', 'https://www.creativedesign.com.bd/api/smsapi');
        $api_key = get_option('cd_sms_api_key');

        if(empty($api_key) || empty($number) || empty($message)) return;

        // Clean phone number (keep only digits)
        $number = preg_replace('/[^0-9]/', '', $number);
        
        // Prepare Data (Sender ID removed)
        $args = array(
            'api_key' => $api_key,
            'type'    => 'text',
            'number'  => $number,
            'message' => $message
        );

        // Build Final URL
        $request_url = add_query_arg($args, $api_url);

        // Send Request using WordPress HTTP API
        $response = wp_remote_get($request_url);

        // Error Logging (Optional)
        if (is_wp_error($response)) {
            error_log('CreativeSMS Error: ' . $response->get_error_message());
        }
    }

    // --- 3. WooCommerce: Admin New Order Notification ---
    public function send_admin_new_order_sms($order_id) {
        if(!get_option('cd_sms_enable_admin')) return;

        $order = wc_get_order($order_id);
        $admin_phone = get_option('cd_sms_admin_phone');
        
        if(!$order || empty($admin_phone)) return;

        $total = $order->get_total();
        $currency = $order->get_currency();
        $customer_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();

        $message = "New Order #{$order_id} Received.\nAmount: {$total} {$currency}\nCustomer: {$customer_name}";

        $this->send_sms($admin_phone, $message);
    }

    // --- 4. WooCommerce: Customer Order Status Notification ---
    public function send_customer_order_status_sms($order_id, $old_status, $new_status, $order) {
        if(!get_option('cd_sms_enable_customer')) return;

        $billing_phone = $order->get_billing_phone();
        if(empty($billing_phone)) return;

        $shop_name = get_bloginfo('name');
        $message = "";

        // Status Messages
        if ($new_status == 'processing') {
            $message = "Hi, Your order #{$order_id} at {$shop_name} is now Processing. Thanks for shopping with us!";
        } 
        elseif ($new_status == 'completed') {
            $message = "Great news! Your order #{$order_id} at {$shop_name} is Completed. Enjoy!";
        }
        elseif ($new_status == 'cancelled') {
             $message = "Order #{$order_id} at {$shop_name} has been Cancelled.";
        }
        elseif ($new_status == 'on-hold') {
             $message = "Your order #{$order_id} at {$shop_name} is currently On Hold.";
        }

        if (!empty($message)) {
            $this->send_sms($billing_phone, $message);
        }
    }

    // --- 5. Registration OTP / Welcome SMS ---
    public function send_registration_otp_sms($user_id) {
        $user_info = get_userdata($user_id);
        
        // Try to get phone number
        $phone = get_user_meta($user_id, 'billing_phone', true);
        
        // Fallback: Check POST data if registering via checkout
        if(empty($phone) && isset($_POST['billing_phone'])) {
            $phone = sanitize_text_field($_POST['billing_phone']);
        }

        if(!empty($phone)) {
            // Generate OTP
            $otp = rand(1000, 9999);
            
            // Save OTP to user meta (for verification plugins)
            update_user_meta($user_id, 'cd_sms_otp', $otp);

            // Get template and replace {otp}
            $template = get_option('cd_sms_otp_template', 'Welcome! Your OTP is {otp}');
            $message = str_replace('{otp}', $otp, $template);

            $this->send_sms($phone, $message);
        }
    }

}

// Initialize Plugin
new CreativeDesign_SMS_Plugin();